﻿
using System;
using System.Collections.Generic;
using System.Linq;
using Optimization.GeneticAppliances.Warehouse;
using Optimization.Helpers;


namespace Optimization.GeneticAlgorithms.Crossovers.ConflictResolvers
{
    class WarehouseMixedNearestNeighborResolver : ConflictResolver
    {
        private double[][] _distancesMatrix;
        private int[] _warehousePointsByLocation;
        private int[] _productsByFrequency;
        Random rnd;
        public WarehouseMixedNearestNeighborResolver(Random random, double probability) : base(random, probability)
        {
            _distancesMatrix = Distances.GetInstance().DistancesMatrix;
            _warehousePointsByLocation = Enumerable.Range(0, _distancesMatrix.Length).OrderBy(x => _distancesMatrix[0][x]).ToArray();
            _productsByFrequency = Enumerable.Range(0, _distancesMatrix.Length).OrderByDescending(x => Orders.ProductFrequency[x]).ToArray();
            rnd = random;
        }


        public override int ResolveConflict(int currentPoint, List<int> availableVertexes)
        {
            int bestCandidate;
            if (rnd.NextDouble() > 0.67)
                bestCandidate = ResolvePairwiseConflict(currentPoint, availableVertexes);
            else
                bestCandidate = ResolveSingleConflict(currentPoint, availableVertexes);
            return bestCandidate;
        }

        private int ResolvePairwiseConflict(int currentPoint, List<int> availableVertexes)
        {
            int pointCount = availableVertexes.Count;

            if (currentPoint == 0) return availableVertexes[0];
            int bestCandidate = availableVertexes[0];
            int bestFrequency = Orders.ProductsTogetherFrequency[currentPoint][bestCandidate];

            for (int i = 1; i < pointCount; i++)
            {
                int frequency = Orders.ProductsTogetherFrequency[currentPoint][availableVertexes[i]];
                if (frequency > bestFrequency)
                {
                    bestCandidate = availableVertexes[i];
                    bestFrequency = frequency;
                }
            }

            return bestCandidate;
        }




        private int ResolveSingleConflict(int currentPoint, List<int> availableVertexes)
        {
            int pointCount = availableVertexes.Count;

            int bestCandidate = availableVertexes[0];
            int currentLocationInChromosome = Orders.ProductFrequency.Length - pointCount - 1;
            int locationIndex = Array.IndexOf(_warehousePointsByLocation, currentLocationInChromosome);
            int bestFit = Math.Abs(Array.IndexOf(_productsByFrequency, availableVertexes[0]) - locationIndex);

            for (int i = 0; i < availableVertexes.Count; i++)
            {
                int candidateIndex = Array.IndexOf(_productsByFrequency, availableVertexes[i]);
                int currentFit = Math.Abs(candidateIndex - locationIndex);
                if (currentFit < bestFit)
                {
                    bestCandidate = availableVertexes[i];
                    bestFit = currentFit;
                }
            }

            return bestCandidate;


        }
    }

}